/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

        Manage PGPKeyDBs

        $Id: pgpKeyDB.c,v 1.81.2.1 2001/04/26 23:17:28 hal Exp $
____________________________________________________________________________*/
#include <stdio.h>
#include <string.h>

#include "pgpConfig.h"
#include "pgpMemoryMgr.h"
#include "pgpOptionListPriv.h"
#include "pgpPktByte.h"
#include "pgpSigSpec.h"
#include "pgpP11Key.h"

#if HAVE_FCNTL_H
#include <fcntl.h>		/* For O_CREAT, O_EXCL */
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifndef W_OK
#define W_OK 2
#endif

#include "pgpKeyPriv.h"
#include "pgpFileRef.h"
#include "pgpFileSpec.h"
#include "pgpFileUtilities.h"
#include "pgpStrings.h"
#include "pgpFileNames.h"
#include "pgpDebug.h"
#include "pgpFile.h"
#include "pgpContext.h"


/* State variable for whether we need to query the backend */
static PGPInt32					sBackendUpdatesNeeded;

struct PGPNotification
{
	struct PGPNotification *	next;			/* Chaining pointer */
	PGPConnectRef				id;				/* Client ID */
	PGPKeySetRef				newSet;			/* Set of new keys */
	PGPKeySetRef				changeSet;		/* Set of changed keys */
};

typedef struct PGPNotification PGPNotification;

static void sRemoveNotification( PGPKeyDB *db );
static void sAddNotification( PGPKeyDB *db );

/***************************** File IO ****************************/



static PGPError
sOpenFile ( PFLFileSpecRef fileRef,
			PGPFileType fileType, PGPFlags openFlags, FILE **stdFile )
{
	PGPError		err;
	PGPFileOpenFlags	flags = kPGPFileOpenReadWritePerm | kPGPFileOpenLockFile;
#if	PGP_UNIX_HPUX || PGP_UNIX_AIX
	/*
	 *	This is needed since the sdk service has not been ported to
	 *	AIX or HP-UX This hack allows E-Biz to open the keyring files
	 *	multiple times in read-only mode.
	 */
	if(!(openFlags & kPGPOpenKeyDBFileOptions_Mutable))
		flags = kPGPFileOpenReadPerm | kPGPFileOpenLockFile;
#endif 	/* PGP_UNIX_HPUX || PGP_UNIX_AIX */
	

	err = pgpFileRefStdIOOpen( fileRef, flags, fileType, stdFile );

	if ( IsPGPError( err ) )
	{
		if ( err == kPGPError_FileNotFound &&
			 				( openFlags & kPGPOpenKeyDBFileOptions_Create ) )
		{
			err = pgpCreateFile( fileRef, fileType );
			if ( IsPGPError( err ) )
				goto cleanup;

			err = pgpFileRefStdIOOpen( fileRef, flags, fileType, stdFile );
		}
		/*
		 *      This is for the case when user wants to open a file readonly
		 *      and the file permissions are set for readonly.
		 */
		else if ( err == kPGPError_FilePermissions
				  && !(openFlags & kPGPOpenKeyDBFileOptions_Mutable)
				  && ((flags & kPGPFileOpenReadWritePerm) ==
					  kPGPFileOpenReadWritePerm ) )
		{
			flags = kPGPFileOpenReadPerm | kPGPFileOpenLockFile;
			err = pgpFileRefStdIOOpen( fileRef, flags, fileType, stdFile );
		}
		if ( IsPGPError( err ) )
			goto cleanup;
	}
 cleanup:
	return err;
}


static PGPError
sOpenKeyRings ( PGPKeyDB *kdb )
{
	PGPError		err;

	if( IsntNull( kdb->privFileRef ) )
	{
		err = sOpenFile( kdb->privFileRef, kPGPFileTypePrivRing,
						 kdb->openFlags, &kdb->privStdFile );
		if( IsPGPError( err ) )
			goto error;
	}
	
	err = sOpenFile( kdb->pubFileRef, kPGPFileTypePubRing, kdb->openFlags,
					 &kdb->pubStdFile );
	if( IsPGPError( err ) )
		goto error;

	if( IsntNull( kdb->privFileRef ) )
	{
		kdb->privFile = pgpFileReadOpen ( kdb->context, kdb->privStdFile,
										  NULL, NULL );
		if ( IsNull( kdb->privFile ) )
		{
			err = kPGPError_OutOfMemory;
			goto error;
		}
	}

	kdb->pubFile = pgpFileReadOpen ( kdb->context, kdb->pubStdFile,
									 NULL, NULL );
	if ( IsNull( kdb->pubFile ) )
	{
		err = kPGPError_OutOfMemory;
		goto error;
	}
	
	if( IsntNull( kdb->privFileRef ) )
	{
		err = pgpReadKeyFile( kdb, kdb->privFile, TRUE );
		if( IsPGPError( err ) )
			goto error;
	}

	err = pgpReadKeyFile( kdb, kdb->pubFile, TRUE );
	if( IsPGPError( err ) )
		goto error;

	/* Check for key being on a hardware token */
	pgpSyncTokenToKeyDB( kdb->context, kdb, FALSE );

	if( kdb->openFlags & kPGPOpenKeyDBFileOptions_Mutable )
		kdb->bmutable = TRUE;

 error:
	if( IsPGPError( err ) )
	{
		/* Clean up on error */
		if( IsntNull( kdb->pubFile ) )
			pgpFileClose( kdb->pubFile );
		else if( IsntNull( kdb->pubStdFile ) )
			pgpStdIOClose( kdb->pubStdFile );
		if( IsntNull( kdb->privFile ) )
			pgpFileClose( kdb->privFile );
		else if( IsntNull( kdb->privStdFile ) )
			pgpStdIOClose( kdb->privStdFile );
		kdb->pubFile = kdb->privFile = NULL;
		kdb->pubStdFile = kdb->privStdFile = NULL;
	}
	return err;
}

static PGPError
sWriteObjTrust( PGPKeyDBObj const *obj, PGPFile *outFile1,
	PGPFile *outFile2 )
{
	PGPByte				pkt[5];
	PGPSize				pktlen;
	PGPKeyInfo *		kinfo;
	PGPUserIDInfo *		uinfo;
	PGPSigInfo *		sinfo;
	PGPCRLInfo *		cinfo;

	pkt[0] = PKTBYTE_BUILD(PKTBYTE_TRUST, 0);
	pktlen = 1;

	switch (pgpObjectType(obj)) {
	case RINGTYPE_KEY:
	case RINGTYPE_SUBKEY:
		kinfo = pgpKeyToKeyInfo( obj );
		pkt[2] = kinfo->trust;
		break;
	case RINGTYPE_USERID:
		/*
		 * Names have 3 bytes of trust.  The first byte
		 * is a 2.x-compatible trust (validity) byte.  The second is
		 * a validity value (how sure are we that this name
		 * is correct - computed), and the third is a confidence
		 * in the named individual as an introducer.
		 */
		uinfo = pgpUserIDToUserIDInfo( obj );
		pkt[2] = uinfo->oldvalidity;
		pkt[3] = uinfo->validity;
		pkt[4] = uinfo->confidence;
		pktlen = 3;
		break;
	case RINGTYPE_SIG:
		/*
		 * 2.x compatibility kludge: Don't write trust
		 * on good compromise certificates.  PGP 2.x
		 * maintenance dies (assert fail) if it finds
		 * trust packets on key sigs.
		 */
		sinfo = pgpSigToSigInfo( obj );
		pkt[2] = sinfo->trust;
		if (OBJISKEY(obj->up) && obj->up == sinfo->by
		    && sinfo->type == PGP_SIGTYPE_KEY_REVOKE)
			return kPGPError_NoErr;
		break;
	case RINGTYPE_CRL:
		cinfo = pgpCRLToCRLInfo( obj );
		pkt[2] = cinfo->trust;
		break;
	  default:
		pgpAssert(0);
	}
	pkt[1] = pktlen;
	pktlen += 2;
	if( IsntNull( outFile1 ) )
	{
		if (pgpFileWrite(pkt, pktlen, outFile1) != pktlen)
			return kPGPError_WriteFailed;
	}
	if( IsntNull( outFile2 ) )
	{
		if (pgpFileWrite(pkt, pktlen, outFile2) != pktlen)
			return kPGPError_WriteFailed;
	}
	return kPGPError_NoErr;
}



/*
 * Write out specified objects.  Secret objs and children go to both files,
 * with the keys in public form.
 */
static PGPError
sWriteObj( PGPKeyDBObj *obj, PGPBoolean secflag, PGPFile *privFile,
	PGPFile *pubFile )
{
	PGPByte const * data;
	PGPSize			datalen;
	PGPBoolean		mustfree = FALSE;
	PGPError err = kPGPError_NoErr;

	data = pgpFetchObject( obj, &datalen );
	if( secflag )
	{
		if( IsntNull( privFile ) )
		{
			if( pgpFileWrite( data, datalen, privFile ) != datalen )
			{
				err = kPGPError_WriteFailed;
			}
		}
	}
	if( IsntPGPError( err ) )
	{
		if( OBJISKEY( obj ) && pgpKeyIsSec( obj ) )
		{
			data = pgpKeyDBObjToPubData( obj, &datalen );
			mustfree = TRUE;
		}
		if( pgpFileWrite( data, datalen, pubFile ) != datalen )
		{
			err = kPGPError_WriteFailed;
		}
	}

	if( IsntPGPError( err ) )
		err = sWriteObjTrust( obj, pubFile, (secflag?privFile:NULL) );

	if( mustfree )
		PGPFreeData( (PGPByte *)data );

	return err;
}


/* Write out new keyring files, overwriting old data */
static PGPError
sWriteKeyRings( PGPKeyDB *kdb )
{
	PGPError		err;
	PGPKeyDBObj *	key;
	PGPKeyDBObj *	child;
	PGPKeyDBObj *	gchild;
	PGPBoolean		secflag;

	/* Close the main files, releasing our lock */
	if( IsntNull( kdb->privFileRef ) )
	{
		err = (kdb->privFile != NULL) ? 
			pgpFileClose( kdb->privFile ) : kPGPError_NoErr;
		if ( IsPGPError( err ) )
			goto cleanup;
		kdb->privFile = NULL;
		kdb->privStdFile = NULL;
	}
	err = (kdb->pubFile != NULL) ? pgpFileClose( kdb->pubFile ) : kPGPError_NoErr;
	if ( IsPGPError( err ) )
		goto cleanup;
	kdb->pubFile = NULL;
	kdb->pubStdFile = NULL;

	/* Re-open keyring files in write mode */
	if( IsntNull( kdb->privFileRef ) )
	{
		err = pgpFileRefStdIOOpen( kdb->privFileRef,
								   kPGPFileOpenStdWriteFlags
									   | kPGPFileOpenLockFile,
								   kPGPFileTypePrivRing,
								   &kdb->privStdFile );
		if ( IsPGPError( err ) )
			goto cleanup;
	}
	err = pgpFileRefStdIOOpen( kdb->pubFileRef,
							   kPGPFileOpenStdWriteFlags
								   | kPGPFileOpenLockFile,
							   kPGPFileTypePubRing,
							   &kdb->pubStdFile );
	if ( IsPGPError( err ) )
		goto cleanup;

	/* Create PGPFile objects for the keyring files */
	if( IsntNull( kdb->privFileRef ) )
	{
		kdb->privFile = pgpFileWriteOpen( kdb->context, kdb->privStdFile,
										  NULL );
		if ( IsNull( kdb->privFile ) )
		{
			pgpStdIOClose( kdb->privStdFile );
			err = kPGPError_OutOfMemory;	/* XXX: Return better error */
			goto cleanup;
		}
	}
	kdb->pubFile = pgpFileWriteOpen( kdb->context, kdb->pubStdFile, NULL );
	if ( IsNull( kdb->pubFile ) )
	{
		pgpStdIOClose( kdb->pubStdFile );
		err = kPGPError_OutOfMemory;	/* XXX: Return better error */
		goto cleanup;
	}

	/* Copy the data to the keyring files */
	for (key = kdb->firstKeyInDB; IsntNull(key); key = key->next)
	{
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		secflag = pgpKeyIsSec( key );
		err = sWriteObj( key, secflag, kdb->privFile, kdb->pubFile );
		if( IsPGPError( err ) )
			goto cleanup;
		for (child = key->down; IsntNull(child); child = child->next)
		{
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if( OBJISCRL(child) && !pgpCRLIsCurrent( child, 0) )
				continue;
			if( OBJISSIG(child) && pgpSigIsSuperceded( child ) )
				continue;
			err = sWriteObj( child, secflag, kdb->privFile, kdb->pubFile );
			if( IsPGPError( err ) )
				goto cleanup;
			for (gchild = child->down; IsntNull(gchild); gchild = gchild->next)
			{
				if( !pgpKeyDBObjIsReal( gchild ) )
					continue;
				if( OBJISCRL(gchild) && !pgpCRLIsCurrent( gchild, 0) )
					continue;
				if( OBJISSIG(gchild) && pgpSigIsSuperceded( gchild ) )
					continue;
				err = sWriteObj( gchild, secflag, kdb->privFile, kdb->pubFile);
				if( IsPGPError( err ) )
					goto cleanup;
			}
		}
	}

	/* Make sure the data is written to disk */
	if( IsntNull( kdb->privFileRef ) )
	{
		err = pgpFileClose( kdb->privFile );
		if ( IsPGPError( err ) )
			goto cleanup;
		kdb->privFile = NULL;
		kdb->privStdFile = NULL;
	}
	err = pgpFileClose( kdb->pubFile );
	if ( IsPGPError( err ) )
		goto cleanup;
	kdb->pubFile = NULL; 
	kdb->pubStdFile = NULL;

	/* Reopen & relock the main keyring files to ensure exclusive access */
	if( IsntNull( kdb->privFileRef ) )
	{
		err = pgpFileRefStdIOOpen( kdb->privFileRef,
								   kPGPFileOpenReadWritePerm |
									   kPGPFileOpenLockFile,
								   kPGPFileTypePrivRing,
								   &kdb->privStdFile );
		if ( IsPGPError( err ) )
			goto cleanup;
		kdb->privFile = pgpFileReadOpen ( kdb->context, kdb->privStdFile,
										  NULL, NULL );
		if ( IsNull( kdb->privFile ) )
		{
			err = kPGPError_OutOfMemory;
			goto cleanup;
		}
	}
	err = pgpFileRefStdIOOpen( kdb->pubFileRef,
							   kPGPFileOpenReadWritePerm |
								   kPGPFileOpenLockFile,
							   kPGPFileTypePubRing,
							   &kdb->pubStdFile );
	if ( IsPGPError( err ) )
		goto cleanup;
	kdb->pubFile = pgpFileReadOpen ( kdb->context, kdb->pubStdFile,
									  NULL, NULL );
	if ( IsNull( kdb->pubFile ) )
	{
		err = kPGPError_OutOfMemory;
		goto cleanup;
	}

cleanup:

	return err;
}


/* Update lists when we change the db.  This may be time consuming so should
 * only be done when we know we are through with changes to the db, not
 * every little object addition or deletion.
 */
	static PGPError
sKeyDBUpdateLists( PGPKeyDBRef keydb )
{
	PGPError err = kPGPError_NoErr;
	PGPError rslt;
	PGPKeySetRef set;
	PGPKeyListRef list;

	PGPValidateKeyDB( keydb );

	for( set = keydb->firstSetInDB; IsntNull( set ); set = set->nextSetInDB )
	{
		for( list = set->firstListInSet; IsntNull( list );
												 list = list->nextListInSet )
		{
			rslt = pgpUpdateList( list );
			if( IsPGPError( rslt ) )
				err = rslt;
		}
	}

	return err;
}




/********************  Exported Functions  ************************/


	PGPBoolean
PGPKeyDBIsMutable (PGPKeyDBRef kdb)
{
	if( !pgpKeyDBIsValid( kdb ) )
		return FALSE;

	pgpEnterBooleanFunction( FALSE );

	return kdb->bmutable;
}


	PGPBoolean
pgpKeyDBIsDirty (PGPKeyDBRef kdb)
{
	pgpa(pgpaPGPKeyDBValid(kdb));

	return kdb->bdirty;
}

	PGPBoolean
pgpKeyDBUsesTokens (PGPKeyDBRef kdb)
{
	pgpa(pgpaPGPKeyDBValid(kdb));

	return PFLFileSpecRefIsValid( kdb->pubFileRef );
}

	void
pgpKeyDBChanged (PGPKeyDBRef kdb, PGPBoolean redoLists)
{
	pgpa(pgpaPGPKeyDBValid(kdb));

	kdb->bdirty = TRUE;
	if( redoLists )
		(void)sKeyDBUpdateLists( kdb );
}

/* Used with pgpContextGetFirstKeyDB to iterate over keydbs */
	PGPKeyDBRef
pgpKeyDBNextKeyDB( PGPKeyDBRef kdb )
{
	pgpa(pgpaPGPKeyDBValid(kdb));
	return kdb->next;
}

	PGPError
pgpKeyDBAddObject_internal (PGPKeyDBRef kdb, PGPKeyDBObj *obj,
	PGPKeyDBObjRef *pnewobj)
{
	PGPKeyDBObj *			parobj;
	PGPKeyDBObj *			newparobj;
	PGPInt32				count;
	PGPInt32				i, j;
	PGPByte const *			objdata;
	PGPSize					objdatalen;
	PGPError				err = kPGPError_NoErr;

	/* New logic goes as follows:
	 * Find keydb of source object
	 * Get data for object
	 * Add object to this keydb
	 * Problem: what if object is a sig?  We need to add parents too
	 * But if we are iterating and adding all objects, it is wasteful
	 * to re-add all parents.
	 * Unless we can check quickly if we have them.
	 * This is similar to logic in initial read.
	 * In fact it might be identical - read each object and its data,
	 * add in self-contained way to keydb.
	 * No, better to read each object's data into a buffer, add that.
	 * So the low level add function takes an object's data, and takes
	 * a parent object, and adds it.  If parent is null then it puts it
	 * in as a top level key.
	 * Problem is this requires us to parse it twice as we are reading
	 * from keyring.  Not really, just have to parse the type of the
	 * packet.
	 */

	/* Count how far we are from the top (0=top, 1=one layer down, etc.) */
	parobj = obj;
	count = 0;
	while( !OBJISTOPKEY(parobj) )
	{
		++count;
		parobj = parobj->up;
	}
	pgpAssert( count <= 2 );

	/* Add object and its parents, starting from the top and working down */
	newparobj = NULL;
	for (i=count; i>=0; --i)
	{
		parobj = obj;
		for (j=0; j<i; ++j)
			parobj = parobj->up;

		/* Check our local cache of parents */
		if( (count-i) < 2  &&  kdb->oldobjs[count-i] == parobj )
		{
			newparobj = kdb->newobjs[count-i];
			continue;
		}

		objdata = pgpFetchObject( parobj, &objdatalen );
		/*
		 * This is the low level add function, also used during keyring
		 * read.  It returns the object we added.  Should be optimized to
		 * be fast for the case where we are adding an already existing
		 * object.
		 */
		newparobj = pgpAddObject(objdata, objdatalen, kdb, newparobj, FALSE,
								 &err);
		if( IsPGPError( err ) )
			goto error;
		pgpAssert( IsntNull( newparobj ) );
		pgpCopyObjectTrust( parobj, newparobj );

		/* Cache our results */
		if( count-i < 2 )
		{
			kdb->oldobjs[count-i]	= parobj;
			kdb->newobjs[count-i]	= newparobj;
		}
	}
	pgpKeyDBChanged( kdb, FALSE );
	if( IsntNull( pnewobj ) )
		*pnewobj = newparobj;
 error:
	return err;
}


	PGPError
pgpKeyDBAddObject (PGPKeyDBRef kdb, PGPKeyDBObj *obj, PGPKeyDBObjRef *pnewobj)
{
	PGPError				err = kPGPError_NoErr;

	if( pgpFrontEndKeyDB( kdb ) )
	{
		PGPUInt32 *newobjs;
		PGPSize newobjslen;
		PGPUInt32 newobjid;

		err = pgpKeyDBAddObject_back( PGPPeekKeyDBContext(kdb), kdb->id,
									  pgpKeyDBObjID(obj), &newobjs,
									  &newobjslen, &newobjid );
		if( IsPGPError( err ) )
			return err;
		err = pgpAddFromKeyArray( kdb, NULL, newobjs, 1, FALSE );
		PGPFreeData( newobjs );
		pgpKeyDBChanged( kdb, FALSE );
		if( IsntNull( pnewobj ) )
			(void)PGPFindNode( kdb->idToObj, newobjid,
							   (PGPUserValue *)pnewobj );
			
		return err;
	}

	return pgpKeyDBAddObject_internal( kdb, obj, pnewobj );
}


	PGPError
pgpKeyDBRemoveObject_internal (PGPKeyDBRef kdb, PGPKeyDBObj *obj)
{
	PGPError		err;

	(void) kdb;
	pgpAssert( PGPPeekKeyDBObjKeyDB(obj) == kdb );

	/* Don't allow deletion of last userid on key */
	if( OBJISUSERID( obj ) )
	{
		PGPKeyDBObj *	parent;
		PGPKeyDBObj *	child;
		PGPUInt32		useridcount;
		useridcount = 0;
		parent = obj->up;
		pgpAssert( OBJISTOPKEY( parent ) );
		for( child=parent->down; IsntNull(child); child=child->next )
		{
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if( OBJISUSERID( obj ) )
			{
				++useridcount;
			}
		}
		if( useridcount < 2 )
			return kPGPError_BadParams;
	}

	/* Don't allow deletion of last self-signature on a userid */
	if( OBJISSIG( obj ) && OBJISUSERID( obj->up ) )
	{
		PGPKeyDBObj *	parent;
		PGPKeyDBObj *	topkey;
		PGPKeyDBObj *	child;
		PGPUInt32		selfsigcount;
		selfsigcount = 0;
		parent = obj->up;
		topkey = parent->up;
		pgpAssert( OBJISUSERID( parent ) );
		pgpAssert( OBJISTOPKEY( topkey ) );
		if( OBJISSIG( obj )
			&& pgpSigMaker( obj ) == topkey )
		{
			for( child=parent->down; IsntNull(child); child=child->next )
			{
				if( !pgpKeyDBObjIsReal( child ) )
					continue;
				if( OBJISSIG( child )
					&& pgpSigMaker( child ) == topkey
					&& (pgpSigType( child ) & 0xf0) == PGP_SIGTYPE_KEY_GENERIC
					&& !pgpSigRevoked( child ) )
				{
					++selfsigcount;
				}
			}
			if( selfsigcount < 2 )
				return kPGPError_DeletingSelfSig;
		}
	}
				
	err = pgpMarkKeyDBObjectDeleted( obj );
	pgpKeyDBChanged ( kdb, FALSE );
	return err;
}

	PGPError
pgpKeyDBRemoveObject (PGPKeyDBRef kdb, PGPKeyDBObj *obj)
{
	PGPError		err;

	(void) kdb;
	pgpAssert( PGPPeekKeyDBObjKeyDB(obj) == kdb );

	if( !pgpKeyDBObjIsReal( obj ) )
		return kPGPError_BadParams;

	if( pgpFrontEndKeyDB( kdb ) )
	{
		err = pgpKeyDBRemoveObject_back( PGPPeekKeyDBContext(kdb), kdb->id,
										 pgpKeyDBObjID(obj) );
		if( IsPGPError( err ) )
			return err;
		pgpMarkKeyDBObjectDeleted( obj );
		pgpKeyDBChanged ( kdb, FALSE );
		return err;
	}

	return pgpKeyDBRemoveObject_internal( kdb, obj );
}


/* True if a front end keydb */
	PGPBoolean
pgpFrontEndKeyDB( PGPKeyDBRef kdb )
{
	return kdb->id != 0;
}


/* Flush any necessary changes out to backing store if present */
	PGPError
pgpKeyDBFlush_internal (PGPKeyDBRef kdb, PGPKeySetRef changedSet)
{
	PGPError        error;
	
	if (!kdb->bmutable)
		return kPGPError_ItemIsReadOnly;

	if (!kdb->bdirty)
		return kPGPError_NoErr;

	/* Recheck sigs before writing out */
	error = pgpCheckKeyRingSigs_internal( kdb->rootSet, NULL, FALSE, 0, 0,
										  changedSet);
	if( IsPGPError( error ) )
		return error;
	error = pgpPropagateTrustInternal( kdb->rootSet, NULL, PGPGetTime(),
									   changedSet );
	if( IsPGPError( error ) )
		return error;

	pgpAssert( IsntNull( kdb->pubFile ) );

	error = sWriteKeyRings (kdb);

	if (error)
		return error;
	kdb->bdirty = 0;

	return kPGPError_NoErr;
}

	static PGPError
pgpKeyDBFlushInternal (PGPKeyDBRef kdb)
{
	PGPError err;

	if( pgpFrontEndKeyDB( kdb ) ) {
		PGPUInt32 *changelist;
		PGPSize changelistsize;

		if( IsPGPError( err = pgpKeyDBFlush_back( PGPPeekKeyDBContext(kdb),
							kdb->id, &changelist, &changelistsize ) ) )
			return err;
		err = pgpKeyRefreshFromList( kdb, changelist, changelistsize );
		kdb->bdirty = 0;
		return err;
	}

	return pgpKeyDBFlush_internal( kdb, NULL );
}




	PGPError
PGPFilterKeyDB( PGPKeyDBRef keyDB, PGPFilterRef filter,PGPKeySetRef *resultSet)
{
	PGPValidateKeyDB( keyDB );

	pgpEnterPGPErrorFunction();

	return PGPFilterKeySet( keyDB->rootSet, filter, resultSet );
}


	PGPBoolean
pgpKeyDBIsValid( PGPKeyDB const *	keyDB)
{
	return( IsntNull( keyDB ) && keyDB->fixedMagic == kPGPKeyDBMagic );
}

#if PGP_DEBUG	/* [ */
	PGPBoolean
pgpaInternalPGPKeyDBValid(
	pgpaCallPrefixDef,
	PGPKeyDB const *	keyDB,
	char const *		varName)
{
	pgpaAddrValid(keyDB, PGPKeyDB);
	pgpaFailIf(keyDB->refCount <= 0, (pgpaFmtPrefix, "refCount <= 0"));
	pgpaFmtMsg((pgpaFmtPrefix,
			"pgpaPGPKeyDBValid failed on %s (%p)", varName, keyDB));

	return pgpaFailed;
}
#endif /* ] PGP_DEBUG */


	PGPError
PGPIncKeyDBRefCount( PGPKeyDBRef kdb )
{
	PGPValidateKeyDB( kdb );

	pgpEnterPGPErrorFunction();

	++kdb->refCount;
	return kPGPError_NoErr;
}


	void
pgpKeyDBDestroy_internal (PGPKeyDBRef kdb)
{
	PGPKeySetRef		set;
	PGPKeyDBRef			kdbprev;
	PGPNotification *	nt;

	pgpAssertAddrValid( kdb, PGPKeyDB );

	/* First flush out any changes, even if not last copy of keydb */
	if( kdb->bmutable )
	{
		/* Mac does not use front-end/back-end distinction the same way */
#if !PGP_MACINTOSH
		if( !pgpFrontEndKeyDB( kdb ) ) 
#endif
			pgpKeyDBFlushInternal( kdb );
	}

	if( kdb->refCount <= 1 )
	{
		PGPUInt32 id = kdb->id;
		PGPContextRef context = kdb->context;
		
		if( kdb->pubFile )
			pgpFileClose( kdb->pubFile );
		if( kdb->pubFileRef )
			PFLFreeFileSpec( kdb->pubFileRef );
		if( kdb->privFile )
			pgpFileClose( kdb->privFile );
		if( kdb->privFileRef )
			PFLFreeFileSpec( kdb->privFileRef );
		
		for( set=kdb->firstSetInDB; IsntNull(set);  )
		{
			pgpFreeKeySet( set, TRUE );
			/* Freeing keyset fixes up list so we can just stay at beginning */
			set = kdb->firstSetInDB;
		}

		memPoolEmpty( &kdb->objPool );
		memPoolEmpty( &kdb->structPool );
		memPoolEmpty( &kdb->pathpool );
		memPoolEmpty( &kdb->regexps );

		PGPDisposeBinaryTree( kdb->idToObj );

		/* Unlink kdb from context list */
		kdbprev = pgpContextGetFirstKeyDB( kdb->context );
		if( kdbprev == kdb )
		{
			pgpContextSetFirstKeyDB( kdb->context, kdb->next );
		} else {
			while( kdbprev->next != kdb )
				kdbprev = kdbprev->next;
			kdbprev->next = kdb->next;
		}

		nt = kdb->notifies;
		while( IsntNull( nt ) )
		{
			PGPNotification *ntnext = nt->next;
			PGPFreeData( nt );
			nt = ntnext;
		}

		PGPFreeData( kdb );

		if( id != 0 )
		{
			pgpFreeKeyDB_back( context, id );
		}

	} else {
		if( !pgpFrontEndKeyDB( kdb ) )
			sRemoveNotification( kdb );

		--kdb->refCount;
	}
	return;
}

	static void
pgpKeyDBDestroyInternal (PGPKeyDBRef kdb)
{
	pgpKeyDBDestroy_internal( kdb );
}


	PGPError
PGPFreeKeyDB(PGPKeyDBRef kdb)
{
	PGPValidateKeyDB( kdb );

	pgpEnterPGPErrorFunction();

	pgpKeyDBDestroyInternal( kdb );
	return kPGPError_NoErr;
}


static void
pgpKeyDBInit( PGPContextRef context, PGPKeyDB *kdb )
{
	pgpClearMemory( kdb, sizeof(*kdb) );

	kdb->fixedMagic = kPGPKeyDBMagic;

	kdb->context	= context;
	kdb->refCount	= 1;
	
	memPoolInit( context, &kdb->objPool);
	memPoolInit( context, &kdb->structPool );
	memPoolInit( context, &kdb->pathpool );
	memPoolInit( context, &kdb->regexps );

	PGPNewBinaryTree( PGPPeekContextMemoryMgr(context), &kdb->idToObj );

	kdb->rootSet = pgpRootSet( kdb );

	/* Link kdb into context list */
	kdb->next = pgpContextGetFirstKeyDB( context );
	pgpContextSetFirstKeyDB( context, kdb );

	/* Register for notifications if we are backend */
	sAddNotification( kdb );
}

	PGPKeySet *
pgpKeyDBPeekRootSet( PGPKeyDB *kdb )
{
	pgpa(pgpaPGPKeyDBValid(kdb));

	return kdb->rootSet;
}

	PGPKeySetRef
PGPPeekKeyDBRootKeySet( PGPKeyDBRef keyDB )
{
	if( ! pgpKeyDBIsValid( keyDB ) )
		return kInvalidPGPKeySetRef;
		
	pgpEnterZeroFunction();

	return pgpKeyDBPeekRootSet( keyDB );
}

	PGPError
pgpKeyDBError( PGPKeyDB *kdb )
{
	pgpa(pgpaPGPKeyDBValid(kdb));
	return kdb->err;
}

	void
pgpKeyDBSetError( PGPKeyDB *kdb, PGPError err )
{
	pgpa(pgpaPGPKeyDBValid(kdb));
	kdb->err = err;
}


	PGPError
PGPFlushKeyDB(PGPKeyDBRef keydb)
{
	PGPValidateKeyDB( keydb );

	pgpEnterPGPErrorFunction();

	return pgpKeyDBFlushInternal( keydb );
}


	PGPError
PGPCountKeysInKeyDB(
	PGPKeyDBRef 	keydb,
	PGPUInt32 *		numKeys )
{
	PGPKeyDBObj *	key;
	PGPUInt32		count = 0;
	PGPError		err	= kPGPError_NoErr;
	
	PGPValidatePtr( numKeys );
	*numKeys	= 0;
	PGPValidateKeyDB( keydb );

	pgpEnterPGPErrorFunction();

	for (key = keydb->firstKeyInDB; key; key = key->next)
	{
		if( pgpKeyDBObjIsReal(key) )
			count++;
	}
	 
	*numKeys = count;

	return( err );
}


	static PGPError
sMatchingKeyDB (PGPContextRef context,
	PFLFileSpecRef pubFileRef, PFLFileSpecRef privFileRef,
	PGPKeyDBRef *keydb )
{
	PGPKeyDBRef		kdb;
	PGPBoolean		match = FALSE;

	*keydb = NULL;

	for( kdb=pgpContextGetFirstKeyDB(context); IsntNull(kdb); kdb=kdb->next )
	{
		if( pgpFrontEndKeyDB( kdb ) )
			continue;

		if( IsNull( kdb->privFileRef )  &&  IsNull( kdb->pubFileRef ) )
			continue;

		if( IsNull(kdb->privFileRef) || IsNull(privFileRef) )
			match = (kdb->privFileRef == privFileRef);
		else
			match = PFLFileSpecsEqual( kdb->privFileRef, privFileRef );
		match &= PFLFileSpecsEqual( kdb->pubFileRef, pubFileRef );

		if( match )
			break;
	}

	if( match )
	{
		pgpAssert( IsntNull( kdb ) );
		++kdb->refCount;
		*keydb = kdb;
	}

	return kPGPError_NoErr;
}


/****************************  Notifications  **************************/

	void
pgpSetPendingNotify( PGPKeySetRef changeset, PGPKeyDBObjRef changeobj,
	PGPKeySetRef newset, PGPKeyDBObjRef newobj )
{
	PGPKeyDB *db = NULL;
	PGPContextRef context;
	PGPNotification *nt;
	PGPConnectRef id = kPGPConnectRef_Null;

	if( IsntNull( changeset ) || IsntNull( newset ) )
	{
		db = PGPPeekKeySetKeyDB( IsntNull(changeset)?changeset:newset );
	} else {
		pgpAssert( IsntNull(changeobj) || IsntNull(newobj) );
		db = PGPPeekKeyDBObjKeyDB( IsntNull(changeobj)?changeobj:newobj );
	}
	context = PGPPeekKeyDBContext( db );
	id = pgpContextGetConnectRef( context );
	for( nt = db->notifies; IsntNull( nt ); nt = nt->next )
	{
		if( pgpConnectRefEqual( nt->id, id ) )
			continue;
		if( IsntNull( changeset ) )
			PGPAddKeys( changeset, nt->changeSet );
		if( IsntNull( newset ) )
			PGPAddKeys( newset, nt->newSet );
		if( IsntNull( changeobj ) )
			PGPAddKey( changeobj, nt->changeSet );
		if( IsntNull( newobj ) )
			PGPAddKey( newobj, nt->newSet );
	}

#if PGP_WIN32
/* Add support for other hosts that do notification */
	if( IsntNull( db ) )
	{
		for( nt = db->notifies; IsntNull( nt ); nt = nt->next )
		{
			if( pgpConnectRefEqual( nt->id, id ) )
				continue;
			if( ( !pgpKeySetIsEmpty( nt->changeSet ) )
				|| !pgpKeySetIsEmpty( nt->newSet ) )
			{
				pgpNotifyClient( nt->id, 0L );
			}
		}
	}
#endif
}

	PGPError
pgpUpdateKeyDB_internal( PGPKeyDB *db, PGPKeySetRef newSet,
						 PGPKeySetRef changeSet )
{
	PGPNotification *nt;
	PGPContextRef context;
	PGPConnectRef id;
	PGPError err = kPGPError_NoErr;

	context = PGPPeekKeyDBContext( db );
	id = pgpContextGetConnectRef( context );

	for( nt = db->notifies; IsntNull( nt ); nt = nt->next )
	{
		if( pgpConnectRefEqual( nt->id, id ) )
		{
			err = PGPAddKeys( nt->newSet, newSet );
			if( IsPGPError( err ) )
				break;
			err = PGPAddKeys( nt->changeSet, changeSet );
			if( IsPGPError( err ) )
				break;
			PGPFreeKeySet( nt->newSet );
			PGPFreeKeySet( nt->changeSet );
			nt->newSet = nt->changeSet = NULL;
			err = PGPNewEmptyKeySet( db, &nt->newSet );
			if( IsPGPError( err ) )
				break;
			err = PGPNewEmptyKeySet( db, &nt->changeSet );
			if( IsPGPError( err ) )
				break;
			break;
		}
	}
	return err;
}

	void
pgpSetBackendUpdateNeeded()
{
	PGPContextRef	ctx;

	ctx = NULL;
	while( IsntNull( ctx = pgpContextNextContext( ctx ) ) )
	{
		pgpContextSetUpdateNeeded( ctx, TRUE );
		++sBackendUpdatesNeeded;
	}
}


	void
pgpKeyDBBackendUpdate( )
{
	PGPContextRef	ctx;
	PGPKeyDBRef		kdb;
	PGPError		err;
	PGPUInt32		numNewKeys;
	PGPUInt32 *		newKeyArray;
	PGPSize			newKeyArraySize;
	PGPUInt32 *		changelist;
	PGPSize			changelistsize;

	if (!pgpRPCEnabled())
		return;
	if( sBackendUpdatesNeeded > 0 )
	{
		ctx = NULL;
		while( IsntNull( ctx = pgpContextNextContext( ctx ) ) &&
			   pgpContextGetUpdateNeeded( ctx ) )
		{
#if PGP_WIN32
			/* Only handle contexts associated with current thread */
			if( GetCurrentThreadId() != pgpContextGetThreadID(ctx) )
				continue;
#endif
			for( kdb=pgpContextGetFirstKeyDB(ctx); IsntNull(kdb);
				 								kdb=kdb->next )
			{
				err = pgpUpdateKeyDB_back( ctx, kdb->id, &numNewKeys,
										   &newKeyArray, &newKeyArraySize,
										   &changelist, &changelistsize );
				if( IsPGPError( err ) )
					continue;
				err = pgpAddFromKeyArray( kdb, NULL, newKeyArray, numNewKeys,
										  FALSE );
				if( IsntNull( newKeyArray ) )
					PGPFreeData( newKeyArray );
				err = pgpKeyRefreshFromList( kdb, changelist, changelistsize );
			}
			pgpContextSetUpdateNeeded( ctx, FALSE );
			--sBackendUpdatesNeeded;
		}
	}
}

/* External function to poll for whether keydb has been changed in backend */
/* Intended for use on architectures where callbacks can't be done */
	PGPError
PGPKeyDBIsUpdated( PGPKeyDBRef kdb, PGPBoolean *updated )
{
	PGPError		err;
	PGPUInt32		numNewKeys;
	PGPUInt32 *		newKeyArray;
	PGPSize			newKeyArraySize;
	PGPUInt32 *		changelist;
	PGPSize			changelistsize;
	PGPContextRef	context;

	PGPValidatePtr( updated );
	*updated	= FALSE;
	PGPValidateKeyDB( kdb );

	pgpEnterPGPErrorFunction();

	context = kdb->context;
	err = pgpUpdateKeyDB_back( context, kdb->id, &numNewKeys,
							   &newKeyArray, &newKeyArraySize,
							   &changelist, &changelistsize );
	if( IsPGPError( err ) )
		return err;

	/* Return with updated as false if nothing to do */
	if( newKeyArraySize == 0 && changelistsize == 0 )
	{
		if( IsntNull( newKeyArray ) )
			PGPFreeData( newKeyArray );
		if( IsntNull( changelist ) )
			PGPFreeData( changelist );
		return kPGPError_NoErr;
	}

	err = pgpAddFromKeyArray( kdb, NULL, newKeyArray, numNewKeys,
							  FALSE );
	if( IsntNull( newKeyArray ) )
		PGPFreeData( newKeyArray );
	err = pgpKeyRefreshFromList( kdb, changelist, changelistsize );

	if( IsntPGPError( err ) )
		*updated = TRUE;

	return err;
}

	static void
sAddNotification( PGPKeyDB *db )
{
	PGPContextRef context;
	PGPConnectRef id;
	PGPConnectRef nullId = kPGPConnectRef_Null;
	PGPNotification *nt;

	context = PGPPeekKeyDBContext( db );
	id = pgpContextGetConnectRef( context );

	if( !pgpConnectRefEqual( id, nullId ) )
	{
		nt = (PGPNotification *) pgpContextMemAlloc( context, sizeof(*nt), 0 );
		PGPNewEmptyKeySet( db, &nt->newSet );
		PGPNewEmptyKeySet( db, &nt->changeSet );
		nt->id = id;
		nt->next = db->notifies;
		db->notifies = nt;
	}
}

	static void
sRemoveNotification( PGPKeyDB *db )
{
	PGPContextRef context;
	PGPConnectRef id;
	PGPConnectRef nullId = kPGPConnectRef_Null;
	PGPNotification *nt, *ntprev;

	context = PGPPeekKeyDBContext( db );
	id = pgpContextGetConnectRef( context );
	if( pgpConnectRefEqual( id, nullId ) )
		return;

	ntprev = NULL;
	for( nt=db->notifies; IsntNull( nt ); ntprev = nt, nt = nt->next )
	{
		if( pgpConnectRefEqual( nt->id, id ) )
		{
			if( ntprev == NULL )
			{
				db->notifies = nt->next;
			} else {
				ntprev->next = nt->next;
			}
			PGPFreeKeySet( nt->newSet );
			PGPFreeKeySet( nt->changeSet );
			PGPFreeData( nt );
			return;
		}
	}
	pgpAssert( 0 );
}
	


/****************************  Constructors  **************************/

	PGPKeyDBRef 
pgpCreateKeyDBFromKeyRings (PGPContextRef context,
	PFLFileSpecRef pubFileRef, PFLFileSpecRef privFileRef,
	PGPFlags openFlags, PGPError *error)
{
	PGPKeyDBRef    kdb;

	*error = kPGPError_NoErr;

	/* See if we have a matching keydb already */
	*error = sMatchingKeyDB( context, pubFileRef, privFileRef, &kdb );
	if( IsPGPError( *error ) )
		return NULL;

	if( IsntNull( kdb ) )
	{
		sAddNotification( kdb );
		if( openFlags & kPGPOpenKeyDBFileOptions_Mutable )
			kdb->bmutable = TRUE;
		return kdb;
	}

	kdb = (PGPKeyDB *)pgpContextMemAlloc( context, sizeof(PGPKeyDB),
										  kPGPMemoryMgrFlags_Clear);
	if (kdb == NULL)
	{
		*error = kPGPError_OutOfMemory;
		return NULL;
	}

	pgpKeyDBInit( context, kdb );

	if( IsPGPError( *error = PFLCopyFileSpec(pubFileRef, &kdb->pubFileRef) ) )
	{
		pgpKeyDBDestroyInternal( kdb );
		return NULL;
	}
	if(	IsntNull( privFileRef ) &&
		IsPGPError( *error = PFLCopyFileSpec(privFileRef, &kdb->privFileRef )))
	{
		pgpKeyDBDestroyInternal( kdb );
		return NULL;
	}
	kdb->openFlags = openFlags;
	*error = sOpenKeyRings ( kdb );
	if (*error)
	{
		pgpKeyDBDestroyInternal( kdb );
		return NULL;
	}

	return kdb;
}

	PGPKeyDBRef 
pgpCreateKeyDBFromMemory (PGPContextRef context,
	void *buf, PGPSize buflen, PGPError *error)
{
	PGPKeyDBRef		kdb;

	*error = kPGPError_NoErr;
	kdb = (PGPKeyDB *)pgpContextMemAlloc( context, sizeof(PGPKeyDB),
										  kPGPMemoryMgrFlags_Clear);
	if (kdb == NULL)
	{
		*error = kPGPError_OutOfMemory;
		return NULL;
	}

	pgpKeyDBInit( context, kdb );

	/* Memfile keydb's are immutable */
	kdb->bmutable = FALSE;

	kdb->pubFile = pgpFileMemOpen ( context, buf, buflen );
	if (!kdb->pubFile)
	{
		pgpKeyDBDestroyInternal( kdb );
		*error = kPGPError_OutOfMemory;
		return NULL;
	}

	*error = pgpReadKeyFile( kdb, kdb->pubFile, FALSE );
	if (*error)
	{
		pgpKeyDBDestroyInternal( kdb );
		return NULL;
	}

	return kdb;
}


/* Create a new, empty keydb, of the memory type */
	PGPError
pgpNewKeyDB_internal( PGPContextRef context, PGPKeyDBRef *keydb )
{
	PGPKeyDBRef		kdb;

	*keydb	= NULL;

	kdb = (PGPKeyDB *)pgpContextMemAlloc( context, sizeof(PGPKeyDB),
										  kPGPMemoryMgrFlags_Clear);
	if (kdb == NULL)
		return kPGPError_OutOfMemory;

	pgpKeyDBInit( context, kdb );

	/* Memfile keydb's are immutable */
	kdb->bmutable = FALSE;

	*keydb = kdb;
	return kPGPError_NoErr;
}


/* Create a new, empty keydb, of the memory type */
	PGPError
PGPNewKeyDB( PGPContextRef context, PGPKeyDBRef *keydb )
{
	PGPUInt32		kdbid;
	PGPError		err;

	PGPValidatePtr( keydb );
	*keydb	= NULL;
	PGPValidateContext( context );
	
	pgpEnterPGPErrorFunction();

	/* In production mode, if no server, don't use double data structs */
#if !PGP_FORCEBACKEND
	if( !pgpRPCEnabled() )
	{
		return pgpNewKeyDB_internal( context, keydb );
	}
#endif

	err = pgpNewKeyDB_back( context, &kdbid );
	if( IsPGPError( err ) )
		return err;

	return pgpNewFrontEndKeyDB( context,  kdbid, NULL, 0, keydb );
}


/* Create a new keydb for the front end, from the given key array */
	PGPError
pgpNewFrontEndKeyDB( PGPContextRef context, PGPUInt32 kdbid,
	PGPUInt32 *keyArray, PGPUInt32 keyCount, PGPKeyDBRef *keydb )
{
	PGPKeyDB *kdb;
	PGPError err = kPGPError_NoErr;

	PGPValidatePtr( keydb );
	*keydb	= NULL;
	PGPValidateContext( context );
	
	kdb = (PGPKeyDB *)pgpContextMemAlloc( context, sizeof(PGPKeyDB),
										  kPGPMemoryMgrFlags_Clear);
	if (kdb == NULL)
		return kPGPError_OutOfMemory;

	pgpKeyDBInit( context, kdb );
	kdb->id = kdbid;

	err = pgpAddFromKeyArray( kdb, NULL, keyArray, keyCount, TRUE );
	
	if( IsPGPError( err ) )
		pgpKeyDBDestroyInternal( kdb );
	else
		*keydb = kdb;

	return err;
}
	




/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
